/********************************************
 * Project: EyeView                         
 * File   : "EyeView.c"                     
 * Author : Frank Peters                 
 * Date   : 2000-10-18, 2000-12-4     
 * Notes  : Transmission of camera pictures
 * Modifications: 
 *  2001-11-23, T. Strutz: optional Colour Correction
 ********************************************/


#include "EyeView.h"
#include <math.h>

int lut_r[256], lut_g[256], lut_b[256];

// Gamma correction using y(x) = x^0.6 * 255^0.4 
float gamma06_tab[256]={
0, 9.1752, 13.907, 17.7374, 21.0791, 24.0989, 26.8848, 29.49, 31.9499, 
34.2895, 36.5272, 38.6769, 40.7497, 42.7545, 44.6985, 46.5877, 48.4271, 
50.2210, 51.9732, 53.6869, 55.3648, 57.0096, 58.6232, 60.2078, 61.7651, 
63.2966, 64.8037, 66.2879, 67.7502, 69.1918, 70.6137, 72.0167, 73.4017, 
74.7695, 76.1208, 77.4563, 78.7767, 80.0824, 81.3741, 82.6523, 83.9174, 
85.1699, 86.4103, 87.6389, 88.8562, 90.0624, 91.2580, 92.4432, 93.6183, 
94.7837, 95.9396, 97.0864, 98.2241, 99.3531, 100.4737, 101.586, 102.6902, 
103.7865, 104.8752, 105.9564, 107.0303, 108.0971, 109.1569, 110.2098, 
111.2562, 112.2959, 113.3294, 114.3565, 115.3776, 116.3926, 117.4018, 
118.4053, 119.4031, 120.3954, 121.3822, 122.3638, 123.3401, 124.3113, 
125.2774, 126.2386, 127.1950, 128.1466, 129.0935, 130.0358, 130.9736, 
131.9069, 132.8358, 133.7604, 134.6808, 135.5970, 136.5091, 137.4171, 
138.3212, 139.2214, 140.1176, 141.0101, 141.8988, 142.7839, 143.6652, 
144.5430, 145.4173, 146.2881, 147.1554, 148.0193, 148.8799, 149.7372, 
150.5912, 151.4420, 152.2896, 153.1341, 153.9755, 154.8139, 155.6492, 
156.4815, 157.3110, 158.1375, 158.9611, 159.7819, 160.5999, 161.4151, 
162.2276, 163.0374, 163.8445, 164.6490, 165.4509, 166.2502, 167.0469, 
167.8411, 168.6328, 169.4220, 170.2088, 170.9932, 171.7752, 172.5548, 
173.3321, 174.1070, 174.8797, 175.6501, 176.4183, 177.1842, 177.9479, 
178.7095, 179.4688, 180.2261, 180.9812, 181.7343, 182.4853, 183.2342, 
183.9811, 184.7259, 185.4688, 186.2097, 186.9486, 187.6856, 188.4207, 
189.1538, 189.8851, 190.6145, 191.3420, 192.0677, 192.7916, 193.5136, 
194.2339, 194.9524, 195.6692, 196.3842, 197.0974, 197.8090, 198.5188, 
199.2270, 199.9334, 200.6383, 201.3414, 202.0430, 202.7429, 203.4412, 
204.1379, 204.8331, 205.5266, 206.2186, 206.9091, 207.5980, 208.2854, 
208.9713, 209.6557, 210.3387, 211.0201, 211.7001, 212.3786, 213.0557, 
213.7314, 214.4056, 215.0784, 215.7498, 216.4199, 217.0885, 217.7558, 
218.4217, 219.0863, 219.7495, 220.4114, 221.0720, 221.7312, 222.3892, 
223.0459, 223.7012, 224.3553, 225.0082, 225.6597, 226.3101, 226.9591, 
227.6070, 228.2536, 228.8990, 229.5431, 230.1861, 230.8279, 231.4685, 
232.1079, 232.7462, 233.3832, 234.0192, 234.6539, 235.2876, 235.9201, 
236.5514, 237.1817, 237.8108, 238.4388, 239.0658, 239.6916, 240.3163, 
240.9400, 241.5626, 242.1841, 242.8045, 243.4239, 244.0423, 244.6596, 
245.2759, 245.8911, 246.5053, 247.1185, 247.7307, 248.3419, 248.9521, 
249.5612, 250.1694, 250.7766, 251.3829, 251.9881, 252.5924, 253.1957, 
253.7981, 254.3995, 255.
};


/*---------------------------------------------------------------------
 *
 * GetAndSendPict2()
 *    modifies grabbed images using LookUpTables,
 *---------------------------------------------------------------------*/
int GetAndSendPict2( int SerialPort, int ConnectorMode, int resolution)
{
	colimage img;
	int     x, y;
	
	// get picture
	CAMGetColFrame ( &img, 0);

	// send picture
	if  ( ConnectorMode == radio)
	{
		WL_SEND( SerialPort);
//        OSWait( 2);              // wait 2 ms after switch on 
	}

	for ( x = 0; x < 82; x++)
	{
		for ( y = 0; y < 62; y++)
		{
			img[y][x][0] = lut_r[ img[y][x][0] ];
			img[y][x][1] = lut_g[ img[y][x][1] ];
			img[y][x][2] = lut_b[ img[y][x][2] ];
		}
	}

	send_preface( SerialPort);   
	send_colorimage( &img, SerialPort, resolution);
	
	OSWait ( 20);
	if  (ConnectorMode == radio) WL_RECV( SerialPort);
		
	return 1;
}


int GetAndSendPict( int SerialPort, int ConnectorMode, int resolution)
{
	colimage img;

	// get picture
	CAMGetColFrame ( &img, 0);

	// send picture
	if  ( ConnectorMode == radio)
	{
		WL_SEND( SerialPort);
		OSWait( 2);              // wait 2 ms after switch on 
	}

	send_preface( SerialPort);   
	send_colorimage( &img, SerialPort, resolution);
	
	OSWait ( 20);
	if  (ConnectorMode == radio) WL_RECV( SerialPort);
		
	return 1;
}



int ChangeResolution (int *resolution)
{
	if ((*resolution) == res_12bit)
	{
		(*resolution) = res_24bit;
		LCDPrintf( "\nRes. = 24Bit\n");
	}
	else if ((*resolution) == res_24bit)
	{
		(*resolution) = res_12bit;
		LCDPrintf( "\nRes. = 12Bit\n");
	}
	else
		LCDPrintf( "\nRes. unknown %d\n", (*resolution));
	
	return 1;   
}


void print_serialname (int port) 
{
	switch (port)
	{
		case SERIAL1:  LCDPrintf( "SERIAL1");  break;
		case SERIAL2:  LCDPrintf( "SERIAL2");  break;
		case SERIAL3:  LCDPrintf( "SERIAL3");  break;
	}
}


void print_speed (int speed) 
{
	switch (speed)
	{
		case SER9600:   LCDPrintf( "   9600");  break;
		case SER19200:  LCDPrintf( "  19200");  break;
		case SER38400:  LCDPrintf( "  38400");  break;
		case SER115200: LCDPrintf( " 115200");  break;
	}
}


void print_mode (int mode) 
{
	switch (mode)
	{
		case cable:  LCDPrintf( "  cable");  break;
		case radio:  LCDPrintf( "  radio");  break;
		case irda:   LCDPrintf( " IRmate");  break;
	}
}

/*---------------------------------------------------------------------
 *
 * AdjustColour()
 *    enables colour correction,
 *    during continuous display mode:
 *      hold a white sheet of paper in front of the camera and 
 *      press the Key "Adj",
 *      use Key "NoA" to switch off the colour correction
 *---------------------------------------------------------------------*/
int AdjustColour( void)
{
		int x, y;
		float   scale_r, scale_g, scale_b;
		unsigned long r=0, g=0, b=0;
		colimage img;

		// get pictures for autobrightness
		LCDPrintf("wait...\n");
		CAMGetColFrame ( &img, 0);
		OSWait ( 10);
		CAMGetColFrame ( &img, 0);
		
		// get 'mean' colour
		for ( x = 0; x < 82; x+=2)
		{
		  for ( y = 0; y < 62; y+=2)
		  {
			r += img[y][x][0];
			g += img[y][x][1];
			b += img[y][x][2];
		  }
		}
		g = (r + g + b) / 3;
		// with simple contrast enhancement for dark image regions
		scale_r = (float)g / (float)r;
		scale_g = (float)g / (float)g;
		scale_b = (float)g / (float)b;
		
		for ( x = 0; x < 256; x++)
		{
			y = (int)floor( gamma06_tab[x]* scale_r);
			if (y < 256)    lut_r[x] = (unsigned char)y;
			else            lut_r[x] = 255;
			y = (int)floor( gamma06_tab[x] * scale_g);
			if (y < 256)    lut_g[x] = (unsigned char)y;
			else            lut_g[x] = 255;
			y = (int)floor( gamma06_tab[x] * scale_b);
			if (y < 256)    lut_b[x] = (unsigned char)y;
			else            lut_b[x] = 255;
		}
		return 1;
}

int main ()
{
	char    z, *title="EyeView V1.21";              
	int     CamType;    
	int     PictureNumber = 1;
	int     adjust_flag = 0;
	
	// Defaults
	int     SerialPort    = SERIAL1;            // wireless = SERIAL2
	int     SerialSpeed   = SER115200;
	int     ConnectorMode = cable;
	int     resolution    = res_24bit;          // 0 = 12 Bit   ;  1 = 24 Bit


	// clear display and write message  
	LCDClear();
	LCDMode( SCROLLING|NOCURSOR);
	LCDPrintf( "%s\n", title);
	LCDPrintf( "by Frank Peters\n");
	LCDPrintf( "mod.by T.Strutz\n");
	
	LCDPrintf( "Port:\n"); 
	LCDSetPos( 3, 7); print_serialname( SerialPort); LCDPrintf( "\n");
	LCDPrintf( "Speed:\n"); 
	LCDSetPos( 4, 7); print_mode( ConnectorMode); LCDPrintf( "\n");
	LCDPrintf( "Mode:\n"); 
	LCDSetPos( 5, 7); print_speed( SerialSpeed); LCDPrintf( "\n");
	LCDMenu( "Prt", "Mod", "Spd", "OK ");
	
	// set serial port and speed    
	z = KEYRead();
	while (z != KEY4)
	{
		switch (z)
		{
			case KEY1:  
						switch (SerialPort)
						{
							case SERIAL1:   SerialPort = SERIAL2;  break;
							case SERIAL2:   SerialPort = SERIAL3;  break;
							case SERIAL3:   SerialPort = SERIAL1;  break;
						}
						LCDSetPos( 3, 7); print_serialname( SerialPort); LCDPrintf( "\n");   
						break;
			case KEY2:  
						switch (ConnectorMode)
						{
							case cable:  ConnectorMode = radio;  break;
							case radio:  ConnectorMode = irda;   break;
							case irda:   ConnectorMode = cable;  break;
						}
						LCDSetPos( 4, 7); print_mode( ConnectorMode); LCDPrintf( "\n");
						break;
			case KEY3:  
						switch (SerialSpeed)
						{
							case SER9600:  
											SerialSpeed = SER19200;
											break;
							case SER19200:  
											SerialSpeed = SER38400;
											if (ConnectorMode == irda)
												SerialSpeed = SER115200;
											break;
							case SER38400:  
											SerialSpeed = SER115200;
											if (ConnectorMode == radio)
												SerialSpeed = SER9600;
											break;
							case SER115200:  
											SerialSpeed = SER9600;
											break;
						}
						LCDSetPos( 5, 7); print_speed( SerialSpeed); LCDPrintf( "\n");
						break;
		}
		z = KEYRead();
	}

	OSInitRS232( SerialSpeed, NONE, SerialPort); 
	
	if (ConnectorMode == irda)  IrDA_SetSpeed( SerialPort, SerialSpeed);

	// init menu 
	LCDClear();
	LCDMode( SCROLLING|NOCURSOR);
	
	// init camara
	CamType = CAMInit( WIDE);
	LCDPrintf( "\nCamType: %d\n", CamType);
	
	/*
	 * Colour Correction
	 */
	LCDPrintf( "%s\n", title);
	LCDPrintf( "\nAdjust Colour ?\n");
	LCDPrintf( "(use white sheet\n");
	
	LCDMenu( "Yes", "Yes", "Yes", "No");
	z = KEYGet();
	if (z != KEY4)
	{
		adjust_flag = AdjustColour();
	} 
	
	LCDClear();
	LCDMode( SCROLLING|NOCURSOR);
	LCDPrintf( "%s\n", title);
	LCDMenu( "One", "Con", "Res", "END");

	// go
	z = KEYRead();
	while (  z != KEY4  )
	{
		switch (z)
		{
			case KEY1:
						LCDPrintf( "\nsend pict. %d\n",PictureNumber);
						PictureNumber++;        
						if (adjust_flag)
							GetAndSendPict2( SerialPort, ConnectorMode, resolution);
						else
							GetAndSendPict( SerialPort, ConnectorMode, resolution);
						LCDPrintf("OK\n");
						break;

			case KEY2:
						LCDMenu( "Adj","NoA","Res","END");
						while (  z != KEY4  )
						{                                   
							// send picture
							LCDPrintf("\nsend pict. %d\n",PictureNumber);
							PictureNumber++;        
							if (adjust_flag)
							   GetAndSendPict2( SerialPort, ConnectorMode, resolution);
							else
							   GetAndSendPict( SerialPort, ConnectorMode, resolution);

							z = KEYRead();
							if (z == KEY3)
								ChangeResolution( &resolution);
							else if (z == KEY1)
							{
								adjust_flag = AdjustColour();
								LCDPrintf( "Adjust On\n");
							}
							else if (z == KEY2)
							{
								adjust_flag = 0;
								LCDPrintf( "Adjust Off\n");
							}
						}                                           
						LCDMenu("One","Con","Res","END");                       
						OSWait (100);
						break;
			case KEY3:
						ChangeResolution (&resolution);
						break;
		}           
		z = KEYRead();
	}
	return(1);
}
